<?php

declare(strict_types=1);

/*
 * This file is part of the box project.
 *
 * (c) Kevin Herrera <kevin@herrera.io>
 *     Théo Fidry <theo.fidry@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace KevinGH\Box\Composer;

use Generator;
use function Humbug\get_contents;
use Humbug\PhpScoper\Whitelist;
use function iterator_to_array;
use KevinGH\Box\Console\DisplayNormalizer;
use function KevinGH\Box\FileSystem\dump_file;
use function KevinGH\Box\FileSystem\mirror;
use KevinGH\Box\Test\FileSystemTestCase;
use PhpParser\Node\Name\FullyQualified;
use function preg_replace;
use RuntimeException;
use Symfony\Component\Finder\Finder;

/**
 * @covers \KevinGH\Box\Composer\ComposerOrchestrator
 */
class ComposerOrchestratorTest extends FileSystemTestCase
{
    private const FIXTURES = __DIR__.'/../../fixtures/composer-dump';
    private const COMPOSER_AUTOLOADER_NAME = 'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05';

    /**
     * @dataProvider provideComposerAutoload
     */
    public function test_it_can_dump_the_autoloader_with_an_empty_composer_json(
        Whitelist $whitelist,
        string $prefix,
        string $expectedAutoloadContents
    ): void {
        dump_file('composer.json', '{}');

        ComposerOrchestrator::dumpAutoload($whitelist, $prefix, false);

        $expectedPaths = [
            'composer.json',
            'vendor/autoload.php',
            'vendor/composer/autoload_classmap.php',
            'vendor/composer/autoload_namespaces.php',
            'vendor/composer/autoload_psr4.php',
            'vendor/composer/autoload_real.php',
            'vendor/composer/autoload_static.php',
            'vendor/composer/ClassLoader.php',
            'vendor/composer/LICENSE',
        ];

        $actualPaths = $this->retrievePaths();

        $this->assertSame($expectedPaths, $actualPaths);

        $actualAutoloadContents = preg_replace(
            '/ComposerAutoloaderInit[a-z\d]{32}/',
            'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
            get_contents($this->tmp.'/vendor/autoload.php')
        );
        $actualAutoloadContents = DisplayNormalizer::removeTrailingSpaces($actualAutoloadContents);

        $this->assertSame($expectedAutoloadContents, $actualAutoloadContents);

        $this->assertSame(
            <<<'PHP'
<?php

// autoload_psr4.php @generated by Composer

$vendorDir = dirname(dirname(__FILE__));
$baseDir = dirname($vendorDir);

return array(
);

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/composer/autoload_psr4.php')
            )
        );
    }

    /**
     * @dataProvider provideComposerAutoload
     */
    public function test_it_cannot_dump_the_autoloader_with_an_invalid_composer_json(
        Whitelist $whitelist,
        string $prefix
    ): void {
        mirror(self::FIXTURES.'/dir000', $this->tmp);

        dump_file('composer.json', '');

        try {
            ComposerOrchestrator::dumpAutoload($whitelist, $prefix, false);

            $this->fail('Expected exception to be thrown.');
        } catch (RuntimeException $exception) {
            $this->assertSame(
                'Could not dump the autoloader.',
                $exception->getMessage()
            );
            $this->assertSame(0, $exception->getCode());
            $this->assertNotNull($exception->getPrevious());

            $this->assertStringContainsString(
                '"./composer.json" does not contain valid JSON',
                $exception->getPrevious()->getMessage()
            );
        }
    }

    public function test_it_can_dump_the_autoloader_with_a_composer_json_with_a_dependency(): void
    {
        mirror(self::FIXTURES.'/dir000', $this->tmp);

        ComposerOrchestrator::dumpAutoload(Whitelist::create(true, true, true), '', false);

        $expectedPaths = [
            'composer.json',
            'vendor/autoload.php',
            'vendor/composer/autoload_classmap.php',
            'vendor/composer/autoload_namespaces.php',
            'vendor/composer/autoload_psr4.php',
            'vendor/composer/autoload_real.php',
            'vendor/composer/autoload_static.php',
            'vendor/composer/ClassLoader.php',
            'vendor/composer/LICENSE',
        ];

        $actualPaths = $this->retrievePaths();

        $this->assertSame($expectedPaths, $actualPaths);

        $this->assertSame(
            <<<'PHP'
<?php

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

return ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05::getLoader();

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/autoload.php')
            )
        );

        $this->assertSame(
            <<<'PHP'
<?php

// autoload_psr4.php @generated by Composer

$vendorDir = dirname(dirname(__FILE__));
$baseDir = dirname($vendorDir);

return array(
);

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/composer/autoload_psr4.php')
            )
        );
    }

    /**
     * @dataProvider provideComposerAutoload
     */
    public function test_it_cannot_dump_the_autoloader_if_the_composer_json_file_is_missing(
        Whitelist $whitelist,
        string $prefix
    ): void {
        try {
            ComposerOrchestrator::dumpAutoload($whitelist, $prefix, false);

            $this->fail('Expected exception to be thrown.');
        } catch (RuntimeException $exception) {
            $this->assertSame(
                'Could not dump the autoloader.',
                $exception->getMessage()
            );
            $this->assertSame(0, $exception->getCode());
            $this->assertNotNull($exception->getPrevious());

            $this->assertStringContainsString(
                'Composer could not find a composer.json file in',
                $exception->getPrevious()->getMessage()
            );
        }
    }

    /**
     * @dataProvider provideComposerAutoload
     */
    public function test_it_can_dump_the_autoloader_with_a_composer_json_lock_and_installed_with_a_dependency(
        Whitelist $whitelist,
        string $prefix,
        string $expectedAutoloadContents
    ): void {
        mirror(self::FIXTURES.'/dir001', $this->tmp);

        ComposerOrchestrator::dumpAutoload($whitelist, $prefix, false);

        // The fact that there is a dependency in the `composer.json` does not change anything to Composer
        $expectedPaths = [
            'composer.json',
            'composer.lock',
            'vendor/autoload.php',
            'vendor/beberlei/assert/composer.json',
            'vendor/beberlei/assert/lib/Assert/Assert.php',
            'vendor/beberlei/assert/lib/Assert/Assertion.php',
            'vendor/beberlei/assert/lib/Assert/AssertionChain.php',
            'vendor/beberlei/assert/lib/Assert/AssertionFailedException.php',
            'vendor/beberlei/assert/lib/Assert/functions.php',
            'vendor/beberlei/assert/lib/Assert/InvalidArgumentException.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertion.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertionException.php',
            'vendor/beberlei/assert/LICENSE',
            'vendor/composer/autoload_classmap.php',
            'vendor/composer/autoload_files.php',
            'vendor/composer/autoload_namespaces.php',
            'vendor/composer/autoload_psr4.php',
            'vendor/composer/autoload_real.php',
            'vendor/composer/autoload_static.php',
            'vendor/composer/ClassLoader.php',
            'vendor/composer/installed.json',
            'vendor/composer/LICENSE',
        ];

        $actualPaths = $this->retrievePaths();

        $this->assertSame($expectedPaths, $actualPaths);

        $this->assertSame(
            $expectedAutoloadContents,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/autoload.php')
            )
        );

        $this->assertSame(
            <<<'PHP'
<?php

// autoload_psr4.php @generated by Composer

$vendorDir = dirname(dirname(__FILE__));
$baseDir = dirname($vendorDir);

return array(
    'Assert\\' => array($vendorDir . '/beberlei/assert/lib/Assert'),
);

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/composer/autoload_psr4.php')
            )
        );
    }

    public function test_it_can_dump_the_autoloader_with_a_composer_json_lock_and_installed_with_a_dev_dependency(): void
    {
        mirror(self::FIXTURES.'/dir003', $this->tmp);

        $composerAutoloaderName = self::COMPOSER_AUTOLOADER_NAME;

        $expectedAutoloadContents = <<<PHP
<?php

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

return $composerAutoloaderName::getLoader();

PHP;

        ComposerOrchestrator::dumpAutoload(
            Whitelist::create(true, true, true),
            '',
            true
        );

        // The fact that there is a dependency in the `composer.json` does not change anything to Composer
        $expectedPaths = [
            'composer.json',
            'composer.lock',
            'vendor/autoload.php',
            'vendor/beberlei/assert/composer.json',
            'vendor/beberlei/assert/lib/Assert/Assert.php',
            'vendor/beberlei/assert/lib/Assert/Assertion.php',
            'vendor/beberlei/assert/lib/Assert/AssertionChain.php',
            'vendor/beberlei/assert/lib/Assert/AssertionFailedException.php',
            'vendor/beberlei/assert/lib/Assert/functions.php',
            'vendor/beberlei/assert/lib/Assert/InvalidArgumentException.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertion.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertionException.php',
            'vendor/beberlei/assert/LICENSE',
            'vendor/composer/autoload_classmap.php',
            'vendor/composer/autoload_namespaces.php',
            'vendor/composer/autoload_psr4.php',
            'vendor/composer/autoload_real.php',
            'vendor/composer/autoload_static.php',
            'vendor/composer/ClassLoader.php',
            'vendor/composer/installed.json',
            'vendor/composer/LICENSE',
        ];

        $actualPaths = $this->retrievePaths();

        $this->assertSame($expectedPaths, $actualPaths);

        $this->assertSame(
            $expectedAutoloadContents,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/autoload.php')
            )
        );

        $this->assertSame(
            <<<'PHP'
<?php

// autoload_psr4.php @generated by Composer

$vendorDir = dirname(dirname(__FILE__));
$baseDir = dirname($vendorDir);

return array(
);

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/composer/autoload_psr4.php')
            )
        );
    }

    /**
     * @dataProvider provideComposerAutoload
     */
    public function test_it_can_dump_the_autoloader_with_a_composer_json_and_lock_with_a_dependency(
        Whitelist $whitelist,
        string $prefix,
        string $expectedAutoloadContents
    ): void {
        mirror(self::FIXTURES.'/dir002', $this->tmp);

        ComposerOrchestrator::dumpAutoload($whitelist, $prefix, false);

        // The fact that there is a dependency in the `composer.json` does not change anything to Composer
        $expectedPaths = [
            'composer.json',
            'composer.lock',
            'vendor/autoload.php',
            'vendor/beberlei/assert/composer.json',
            'vendor/beberlei/assert/lib/Assert/Assert.php',
            'vendor/beberlei/assert/lib/Assert/Assertion.php',
            'vendor/beberlei/assert/lib/Assert/AssertionChain.php',
            'vendor/beberlei/assert/lib/Assert/AssertionFailedException.php',
            'vendor/beberlei/assert/lib/Assert/functions.php',
            'vendor/beberlei/assert/lib/Assert/InvalidArgumentException.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertion.php',
            'vendor/beberlei/assert/lib/Assert/LazyAssertionException.php',
            'vendor/beberlei/assert/LICENSE',
            'vendor/composer/autoload_classmap.php',
            'vendor/composer/autoload_namespaces.php',
            'vendor/composer/autoload_psr4.php',
            'vendor/composer/autoload_real.php',
            'vendor/composer/autoload_static.php',
            'vendor/composer/ClassLoader.php',
            'vendor/composer/LICENSE',
        ];

        $actualPaths = $this->retrievePaths();

        $this->assertSame($expectedPaths, $actualPaths);

        $this->assertSame(
            $expectedAutoloadContents,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/autoload.php')
            )
        );

        $this->assertSame(
            <<<'PHP'
<?php

// autoload_psr4.php @generated by Composer

$vendorDir = dirname(dirname(__FILE__));
$baseDir = dirname($vendorDir);

return array(
);

PHP
            ,
            preg_replace(
                '/ComposerAutoloaderInit[a-z\d]{32}/',
                'ComposerAutoloaderInit80c62b20a4a44fb21e8e102ccb92ff05',
                get_contents($this->tmp.'/vendor/composer/autoload_psr4.php')
            )
        );
    }

    public function provideComposerAutoload(): Generator
    {
        $composerAutoloaderName = self::COMPOSER_AUTOLOADER_NAME;

        yield [
            Whitelist::create(true, true, true),
            '',
            <<<PHP
<?php

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

return $composerAutoloaderName::getLoader();

PHP
        ];

        yield [
            Whitelist::create(true, true, true, 'Acme\Foo'),  // Whitelist is ignored when prefix is empty
            '',
            <<<PHP
<?php

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

return $composerAutoloaderName::getLoader();

PHP
        ];

        yield [
            Whitelist::create(true, true, true),
            '_Box',
            <<<PHP
<?php

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

return $composerAutoloaderName::getLoader();

PHP
        ];

        yield [
            (static function (): Whitelist {
                $whitelist = Whitelist::create(true, true, true, 'Acme\Foo');

                $whitelist->recordWhitelistedClass(
                    new FullyQualified('Acme\Foo'),
                    new FullyQualified('_Box\Acme\Foo')
                );

                return $whitelist;
            })(),
            '_Box',
            <<<PHP
<?php

// @generated by Humbug Box

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

\$loader = $composerAutoloaderName::getLoader();

// Aliases for the whitelisted classes. For more information see:
// https://github.com/humbug/php-scoper/blob/master/README.md#class-whitelisting
class_exists('_Box\Acme\Foo');

return \$loader;

PHP
        ];

        yield [
            (static function (): Whitelist {
                $whitelist = Whitelist::create(true, true, true, 'Acme\Foo');

                $whitelist->recordWhitelistedFunction(
                    new FullyQualified('foo'),
                    new FullyQualified('_Box\foo')
                );

                return $whitelist;
            })(),
            '_Box',
            <<<PHP
<?php

// @generated by Humbug Box

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

\$loader = ${composerAutoloaderName}::getLoader();

// Functions whitelisting. For more information see:
// https://github.com/humbug/php-scoper/blob/master/README.md#functions-whitelisting
if (!function_exists('foo')) {
    function foo() {
        return \_Box\\foo(...func_get_args());
    }
}

return \$loader;

PHP
        ];

        yield [
            (static function (): Whitelist {
                $whitelist = Whitelist::create(true, true, true, 'Acme\Foo');

                $whitelist->recordWhitelistedFunction(
                    new FullyQualified('foo'),
                    new FullyQualified('_Box\foo')
                );
                $whitelist->recordWhitelistedFunction(
                    new FullyQualified('Acme\foo'),
                    new FullyQualified('_Box\Acme\foo')
                );

                return $whitelist;
            })(),
            '_Box',
            <<<PHP
<?php

// @generated by Humbug Box

namespace {

// autoload.php @generated by Composer

require_once __DIR__ . '/composer/autoload_real.php';

\$loader = ${composerAutoloaderName}::getLoader();

}

// Functions whitelisting. For more information see:
// https://github.com/humbug/php-scoper/blob/master/README.md#functions-whitelisting
namespace {
    if (!function_exists('foo')) {
        function foo() {
            return \_Box\\foo(...func_get_args());
        }
    }
}
namespace Acme {
    if (!function_exists('Acme\\foo')) {
        function foo() {
            return \_Box\Acme\\foo(...func_get_args());
        }
    }
}

namespace {
    return \$loader;
}

PHP
        ];
    }

    /**
     * @return string[]
     */
    private function retrievePaths(): array
    {
        $finder = Finder::create()->files()->in($this->tmp);

        return $this->normalizePaths(iterator_to_array($finder, false), true);
    }
}
