[![Build Status](https://travis-ci.org/jeremyharris/cakephp-lazyload.svg?branch=master)](http://travis-ci.org/jeremyharris/cakephp-lazyload)
[![codecov](https://codecov.io/gh/jeremyharris/cakephp-lazyload/branch/master/graph/badge.svg)](https://codecov.io/gh/jeremyharris/cakephp-lazyload)
[![Packagist](https://img.shields.io/packagist/dt/jeremyharris/cakephp-lazyload.svg)](https://packagist.org/packages/jeremyharris/cakephp-lazyload)
[![license](https://img.shields.io/github/license/jeremyharris/cakephp-lazyload.svg)]()

# CakePHP ORM LazyLoad Plugin

This is an association lazy loader for CakePHP ORM entities. It allows you to
lazily load association data by accessessing the property, without using
`contain()` (the eager loader).

## Installation

Requirements

- CakePHP ORM (or the full framework) >= 3.6.x, < 4.0.0
- sloth

`$ composer require jeremyharris/cakephp-lazyload`

For older versions of the CakePHP ORM, please use version `2.0.x`.

## Usage

If you have a base entity, add the trait to get lazy loading across
all of your entities. Or, attach it to a single entity to only lazy
load on that entity:

**src/Model/Entity/User.php**
```php
<?php
namespace App\Model\Entity;

use Cake\ORM\Entity;
use JeremyHarris\LazyLoad\ORM\LazyLoadEntityTrait;

class User extends Entity
{
    use LazyLoadEntityTrait;
}
```

Associations to the Users table can now be lazily loaded from the entity!

### Example

Let's assume that our base entity has the `LazyLoadEntityTrait` and:

```text
Brewery hasMany Beers
Programmer belongsToMany Beers
```

With the lazy loader, all we need is the entity:

```php
<?php
// get an entity, don't worry about contain
$programmer = $this->Programmers->get(1);
```

When accessing an association property (as if the data was eagerly loaded), the
associated data is loaded automatically.

```php
<?php
// beers is lazy loaded
foreach ($programmer->beers as $beer) {
    // brewery is lazy loaded onto $beer
    echo $programmer->name . ' drinks beer ' .  $beer->name . ' from ' . $beer->brewery->name;
}
```

### Using contain in conjunction with the lazy loader

The lazy loader will not overwrite results that are generated by the eager
loader (`contain()`). You can continue to write complex contain conditions and
still take advantage of the lazy loader.

```php
<?php
$programmer = $this->Programmers->get(1, [
    'contain' => [
        'Beers'
    ]
]);

// beers is loaded via the eager loader
$programmer->beers;
// brewery is lazy loaded onto $beer[0]
$programmer->beers[0]->brewery;
```

## Entity method support

Entities with the lazy loader trait support lazy loading using the different
property access methods provided by the Cake ORM:

- Getters: `$programmer->get('beers)`, `$programmer->beers`
- Has: `$programmer->has('beers)`
- Unset: `$programmer->unsetProperty('beers')`

When unsetting a property via `Entity::unsetProperty()`, the property will be
prevented from being lazily loaded in the future for that entity, as it
respects the state in the same way a typical Entity would. If you wish to re-hydrate
an association, you can use `Table::loadInto` as provided by the ORM:

```php
<?php
$programmer = $this->Programmers->get(1);

// beers is lazy loaded
$programmer->beers;
// remove beers from the entity
$programmer->unsetProperty('beers');
// this now returns false
$programmer->has('beers');
// if we want access to beers again, we can manually load it
$programmer = $this->Programmers->loadInto($programmer, ['Beers']);
```

## Testing

Sometimes in tests, we create entities that don't necessarily have tables. When
accessing a property that doesn't exist, the LazyLoad trait will try to load the
table in order to get association data, which would throw an error if the table
doesn't exist. To prevent this, you can override `_repository()` in your entity:

```php
<?php
namespace App\Model\Entity;

use Cake\ORM\Entity;
use Exception;
use JeremyHarris\LazyLoad\ORM\LazyLoadEntityTrait;

class User extends Entity
{
    use LazyLoadEntityTrait {
        _repository as _loadRepository;
    }

    protected function _repository()
    {
        try {
            $repository = $this->_loadRepository();
        } catch (Exception $e) {
            return false;
        }
        return $repository;
    }
}
```

By default, the LazyLoad trait will throw whatever error bubbles up
`TableRegistry::get()`.

### Plugins

If testing plugins entities that don't have tables, make sure to override the
`_repository()` method to return the *plugin's* table.

## Notes

- **Contain:** This is not a replacement for `contain()`, which can write complex queries to dictate
what data to contain. The lazy loader obeys the association's conditions that
you set when defining the association on the table, but apart from that it grabs
all associated data.
- **Speed:** Lazy loading in this manner isn't necessarily a speed improvement. In fact, it can be a
detriment to speed in certain cases, such as looping over entities that lazy load associations within
the loop (creates a single SQL query per-item rather than using joins or the ORM). This plugin is
intended as a helper for bootstrapping projects.
- **Hydration:** The lazy loader requires that your result set is hydrated in order to
provide lazy loading functionality.

> Special thanks to @lorenzo for reviewing the plugin before its initial release!
